#!/usr/bin/env python
#
# server.py  -  Utilities for inspection and service
#
# $Revision$
#
# Copyright (C) 2015 Jan Jockusch <jan.jockusch@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
# $Id$

from time import sleep
from distutils.version import LooseVersion
from perfact.generic import safe_syscall
from perfact.checktarget import checkTCPTarget

PFSYSTEMTYPE_DEVELOPMENT = 'DEVELOPMENT'
PFSYSTEMTYPE_VALIDATION = 'VALIDATION'
PFSYSTEMTYPE_PRODUCTION = 'PRODUCTION'
PFSYSTEMTYPE_UNKNOWN = 'UNKNOWN'


def get_pfsysteminfo(kind='id'):
    assert kind in ['id', 'name', 'type']

    callparam = kind

    # We don't have a file called 'pfsystemtype', so we
    # have to use the name for figuring out the systemtype
    if kind == 'type':
        callparam = 'name'

    retcode, output = safe_syscall(
        ['/bin/cat', '/etc/pfsystem'+callparam])

    if retcode != 0:
        return 'unknown'

    if kind == 'type':
        sysname = output.strip()
        systype = systemtype_from_systemname(sysname)
        return systype
    else:
        return output.strip()


def systemtype_from_systemname(systemname=''):
    if systemname.endswith('-devel') \
       or '-devel-' in systemname:
        return PFSYSTEMTYPE_DEVELOPMENT

    if systemname.endswith('-validation') \
       or '-validation-' in systemname:
        return PFSYSTEMTYPE_VALIDATION

    if systemname.endswith('-prod') \
       or '-prod-' in systemname:
        return PFSYSTEMTYPE_PRODUCTION

    return PFSYSTEMTYPE_UNKNOWN


# Phone home activation, deactivation and status

ph_cmd = '/usr/bin/phonehome'


def phonehome_version():
    retcode, output = safe_syscall([
        'sh', '-c',
        'dpkg -s perfact-phonehome'
        ' | grep "^Version: "'
        ' | sed "s/Version://"'
    ])
    return LooseVersion(output.strip())


def phonehome_status():
    if phonehome_version() >= LooseVersion('2.0.0'):
        retcode, output = safe_syscall([
            'sudo', 'systemctl', 'status', 'perfact-phonehome.service'
        ])
    else:
        retcode, output = safe_syscall([ph_cmd, 'status'])
    return retcode, output


def phonehome_up(num=1):
    if phonehome_version() >= LooseVersion('2.0.0'):
        safe_syscall([
            'sudo', 'systemctl', 'enable', 'perfact-phonehome.service'
        ])
        safe_syscall([
            'sudo', 'systemctl', 'start', 'perfact-phonehome.service'
        ])
    else:
        retcode, output = safe_syscall([ph_cmd, 'up', str(num)])
    return phonehome_status()


def phonehome_down():
    if phonehome_version() >= LooseVersion('2.0.0'):
        safe_syscall([
            'sudo', 'systemctl', 'stop', 'perfact-phonehome.service'
        ])
        safe_syscall([
            'sudo', 'systemctl', 'disable', 'perfact-phonehome.service'
        ])
    else:
        retcode, output = safe_syscall([ph_cmd, 'down'])
    return phonehome_status()


# Printer status

def status_lpstat():
    return safe_syscall('lpstat -p -o')


def status_lpcancel(id=None):
    if id is None:
        return safe_syscall('cancel -a')
    else:
        return safe_syscall('cancel %d' % int(id))


def status_lpinspect(id, page=1):
    filename = '/var/spool/cups/d%05d-%03d' % (int(id), int(page))
    return safe_syscall(['cat', filename], text=False)


def status_lpenable(printer):
    return safe_syscall(['cupsenable', str(printer)])


def status_lpdisable(printer):
    return safe_syscall(['cupsdisable', str(printer)])


def status_lpqueue(printer):
    return safe_syscall(['lpstat', '-W', 'not-completed', str(printer)])


def status_lpqueue_done(printer):
    ret_code, raw_data = safe_syscall(['lpstat', '-W', 'completed',
                                       str(printer)])
    raw_data_lines = raw_data.split('\n')
    result = []
    for item in raw_data_lines:
        if len(item) == 0:
            continue
        parts = item.split(' ')
        job = parts[0]
        ts = '{}-{}-{} {}'.format(parts[-6], parts[-5], parts[-4], parts[-3])
        result.append({'job': job, 'ts': ts})
    return result


# Process listing

def status_top():
    return safe_syscall('top -n1 -b')


def status_process(pid, cmd=None):
    '''Check for the presence of the given PID, returning the command line
    if it matches cmd or cmd is None.

    If the process is not found or does not match the cmd, return
    None.

    '''
    try:
        f = open('/proc/%d/cmdline' % int(pid), 'r')
    except (OSError, IOError):
        return None
    cmdline = f.read().strip()
    f.close()
    if cmd is None or cmdline.find(cmd) != -1:
        return cmdline
    return None


def status_killproc(pid, signal=15):
    retcode, output = safe_syscall(['/usr/local/bin/kill',
                                    '-%d' % signal, '%d' % pid])
    # time.sleep(0.5)
    return retcode, output


# Service restarting

def service_restart(service):
    if service not in ('zope2.13', 'xmitd', 'dbcached', 'sampled'):
        raise ValueError("Invalid service given.")
    return safe_syscall(['sudo', '/usr/sbin/service',
                         service, 'restart'])

# Mail status


def status_postqueue():
    return safe_syscall(['postqueue', '-p'])


def status_postqueue_flush():
    return safe_syscall(['postqueue', '-f'])


def status_postqueue_cancel(id):
    return safe_syscall(['sudo', 'postsuper', '-d', str(id)])


def status_postqueue_inspect(id):
    return safe_syscall(['sudo', 'postcat', '-q', str(id)])


def status_postqueue_fwclose():
    # XXX TODO: This program is not part of this package!
    return safe_syscall([
        'sudo', '/opt/perfact/zope_tools/outgoing_mail_fw.sh', 'start',
    ])
    # return safe_syscall(['postsuper', '-H', str(id)])


def status_postqueue_fwopen():
    # XXX TODO: This program is not part of this package!
    return safe_syscall([
        'sudo', '/opt/perfact/zope_tools/outgoing_mail_fw.sh', 'stop',
    ])


def status_postqueue_fwstatus():
    # XXX TODO: This program is not part of this package!
    return safe_syscall([
        'sudo', '/opt/perfact/zope_tools/outgoing_mail_fw.sh', 'status',
    ])


# Firewall
def status_firewall(host='127.0.0.1', raisemode=False):
    '''Read out the status of the firwall on a given host.
    Returns a nested dictionary whos first level contains only one key/value
    pair with key 'iptables'. Second level has a key for every builtin iptables
    table.
    e.g.: {'iptables': {'filter': '...', 'nat': '...', 'mangle': '...'}}
    The values are the outputs of the 'iptables' command issued on a CLI.
    '''
    from .firewall import firewall_control
    fwstruct = {u'command': None,
                u'fwstruct': {'iptables': {}},
                u'outformat': 'iptables'
                }
    retcode = 0
    output = ''

    try:
        output = firewall_control(host=host, fwstruct=fwstruct)
    except Exception as e:
        if raisemode:
            raise
        retcode = 1
        output = e
    return retcode, output

# Disk stats


def status_disks():
    return safe_syscall('df -h')


# Network stats

def status_netaddr():
    return safe_syscall('ip addr show')


def status_netroute():
    return safe_syscall('ip route show')


# Package versions

def status_pkgversion(package):
    return safe_syscall([
        'dpkg-query', '--showformat=${Version}', '--show', package,
    ])


def status_osversion():
    return safe_syscall(['cat', '/etc/lsb-release'])


# Local server monit page

def status_monit():
    return safe_syscall(['curl', '-s', 'http://localhost:2812'])

# Load balancer status


def status_balancer():
    return safe_syscall([
        'curl', '-s', 'http://localhost:8980/balancer-manager',
    ])


def status_haproxy():
    return safe_syscall([
        'curl', '--insecure', '-s', 'https://localhost/haproxy-stats',
    ])


def status_mailserver(host='127.0.0.1'):
    '''Check status of mailserver. Attempts 5 checks on port 25,
    and if all of them fail, the mailserver is considered down.
    '''
    failures = 0
    for i in range(0, 5):
        status, message = checkTCPTarget(
            target_ip=host,
            target_port=25,  # SMTP port
        )
        if status != 0:
            failures += 1
        sleep(0.25)
    # only return false if all 5 checks failed
    return failures < 5
